import { NextResponse } from "next/server";

// ===========================================
// 型定義
// ===========================================

export interface ApiSuccessResponse<T> {
  success: true;
  data: T;
}

export interface ApiErrorResponse {
  success: false;
  error: {
    code: string;
    message: string;
  };
}

export type ApiResponse<T> = ApiSuccessResponse<T> | ApiErrorResponse;

// ページネーション用
export interface PaginatedData<T> {
  items: T[];
  pagination: {
    page: number;
    perPage: number;
    total: number;
    totalPages: number;
  };
}

// ===========================================
// レスポンスヘルパー関数
// ===========================================

/**
 * 成功レスポンスを返す
 */
export function apiSuccess<T>(data: T, status = 200) {
  return NextResponse.json(
    {
      success: true,
      data,
    } as ApiSuccessResponse<T>,
    { status }
  );
}

/**
 * エラーレスポンスを返す
 */
export function apiError(code: string, message: string, status = 400) {
  return NextResponse.json(
    {
      success: false,
      error: {
        code,
        message,
      },
    } as ApiErrorResponse,
    { status }
  );
}

// ===========================================
// よく使うエラーレスポンス
// ===========================================

export const ApiErrors = {
  // 認証エラー
  unauthorized: () => apiError("UNAUTHORIZED", "認証が必要です", 401),
  forbidden: () => apiError("FORBIDDEN", "アクセス権限がありません", 403),

  // リソースエラー
  notFound: (resource = "リソース") =>
    apiError("NOT_FOUND", `${resource}が見つかりません`, 404),

  // バリデーションエラー
  validation: (message: string) =>
    apiError("VALIDATION_ERROR", message, 400),

  // サーバーエラー
  internal: (message = "サーバーエラーが発生しました") =>
    apiError("INTERNAL_ERROR", message, 500),
};

// ===========================================
// ページネーションヘルパー
// ===========================================

export function paginate<T>(
  items: T[],
  page: number,
  perPage: number,
  total: number
): PaginatedData<T> {
  return {
    items,
    pagination: {
      page,
      perPage,
      total,
      totalPages: Math.ceil(total / perPage),
    },
  };
}
