import { NextRequest } from "next/server";
import { auth } from "@/lib/auth";
import { db } from "@/lib/db";
import { apiSuccess, ApiErrors, paginate } from "@/lib/api-response";

/**
 * GET /api/v1/users
 * ユーザー一覧を取得（認証必須）
 * 
 * Query Parameters:
 * - page: ページ番号（デフォルト: 1）
 * - perPage: 1ページあたりの件数（デフォルト: 10、最大: 100）
 */
export async function GET(req: NextRequest) {
  try {
    // 認証チェック
    const session = await auth();
    if (!session) {
      return ApiErrors.unauthorized();
    }

    // クエリパラメータ取得
    const searchParams = req.nextUrl.searchParams;
    const page = Math.max(1, parseInt(searchParams.get("page") || "1"));
    const perPage = Math.min(100, Math.max(1, parseInt(searchParams.get("perPage") || "10")));
    const skip = (page - 1) * perPage;

    // ユーザー取得
    const [users, total] = await Promise.all([
      db.user.findMany({
        skip,
        take: perPage,
        select: {
          id: true,
          name: true,
          email: true,
          image: true,
          createdAt: true,
        },
        orderBy: {
          createdAt: "desc",
        },
      }),
      db.user.count(),
    ]);

    return apiSuccess(paginate(users, page, perPage, total));
  } catch (error) {
    console.error("GET /api/v1/users error:", error);
    return ApiErrors.internal();
  }
}
